/*
 * odbm - original dbm bridge
 *
 *  (c) 1999 Shiro Kawai  shiro@acm.org
 *  Terms and conditions to use, modify or distribute this code
 *  is the same as of STk.
 *
 *  $Id: odbm.c,v 1.4 2000/02/12 07:08:21 shiro Exp $
 */

#include <stk.h>

#ifdef HAVE_DBM_H
#include <dbm.h>
#elif HAVE_GDBM_DBM_H
#include <gdbm/dbm.h>
#endif

/* Original dbm allows to open only one file at a time.
   The static variable odbm_opened tracks the status.
 */

static int odbm_opened = FALSE;

/* odbm-init NAME
 *  Files NAME.dir and NAME.pag must exist.
 */

PRIMITIVE stk_odbm_init(SCM sname)
{
    int r;
    ENTER_PRIMITIVE("odbm-init");

    if (odbm_opened) Serror("dbm file is already opened", NIL);
    if (!STRINGP(sname)) Serror("string required", sname);
    r = dbminit(CHARS(sname));
    if (r < 0) {
        Serror("couldn't open dbm database", sname);
    }
    odbm_opened = TRUE;
    return STk_makeinteger(r);
}

/* odbm-close
 */

PRIMITIVE stk_odbm_close()
{
    if (odbm_opened) {
        dbmclose();
        odbm_opened = FALSE;
    }
    return UNDEFINED;
}

/* odbm-closed?
 */

PRIMITIVE stk_odbm_closed()
{
    return odbm_opened ? Ntruth : Truth;
}

/* odbm-store key, val
 */

PRIMITIVE stk_odbm_store(SCM skey, SCM sval)
{
    datum key, val;
    int r;
    ENTER_PRIMITIVE("odbm-store");

    if (!odbm_opened) Serror("dbm database is not opened", NIL);
    if (!STRINGP(skey)) Serror("string is required for a key", skey);
    if (!STRINGP(sval)) Serror("string is required for a val", sval);
    key.dptr  = CHARS(skey);
    key.dsize = STRSIZE(skey);
    val.dptr  = CHARS(sval);
    val.dsize = STRSIZE(sval);
    r = store(key, val);
    return STk_makeinteger(r);
}

/* odbm-fetch KEY
 */

PRIMITIVE stk_odbm_fetch(SCM skey)
{
    datum key, val;
    SCM r;
    ENTER_PRIMITIVE("odbm-fetch");
    if (!odbm_opened) Serror("dbm database is not opened", NIL);
    if (!STRINGP(skey)) Serror("string is required for a key", skey);
    key.dptr  = CHARS(skey);
    key.dsize = STRSIZE(skey);
    val = fetch(key);
    if (val.dptr) {
        r = STk_makestrg(val.dsize, val.dptr);
        /* no need to free val.dptr : dbm frees it. */
    } else {
        r = Ntruth;
    }
    return r;
}

/* odbm-delete KEY
 */

PRIMITIVE stk_odbm_delete(SCM skey)
{
    datum key;
    int r;
    ENTER_PRIMITIVE("odbm-delete");
    if (!odbm_opened) Serror("dbm database is not opened", NIL);
    if (!STRINGP(skey)) Serror("string is required for a key", skey);
    key.dptr  = CHARS(skey);
    key.dsize = STRSIZE(skey);
    r = delete(key);
    return STk_makeinteger(r);
}

/* odbm-firstkey
 */

PRIMITIVE stk_odbm_firstkey()
{
    datum key;
    SCM r;
    ENTER_PRIMITIVE("odbm-firstkey");
    if (!odbm_opened) Serror("dbm database is not opened", NIL);
    key = firstkey();
    if (key.dptr) {
        r = STk_makestrg(key.dsize, key.dptr);
        /* no need to free key.dptr : dbm frees it. */
    } else {
        r = Ntruth;
    }
    return r;
}

/* odbm-nextkey KEY
 */

PRIMITIVE stk_odbm_nextkey(SCM skey)
{
    datum key, nkey;
    SCM r;
    ENTER_PRIMITIVE("dbm-nextkey");
    if (!odbm_opened) Serror("dbm database is not opened", NIL);
    if (!STRINGP(skey)) Serror("string is required for a key", skey);
    key.dsize = STRSIZE(skey);
    key.dptr  = CHARS(skey);
    nkey = nextkey(key);
    if (nkey.dptr) {
        r = STk_makestrg(nkey.dsize, nkey.dptr);
        /* no need to free nkey.dptr : dbm frees it. */
    } else {
        r = Ntruth;
    }
    return r;
}

/* Initialization */

#define DEFPRIM(name, tag, proc) \
    STk_add_new_primitive(name, tag, (PRIMITIVE(*)())proc)

void STk_init_odbm()
{
    DEFPRIM("odbm-init", tc_subr_1, stk_odbm_init);
    DEFPRIM("odbm-close", tc_subr_0, stk_odbm_close);
    DEFPRIM("odbm-closed?", tc_subr_0, stk_odbm_closed);
    DEFPRIM("odbm-store", tc_subr_2, stk_odbm_store);
    DEFPRIM("odbm-fetch", tc_subr_1, stk_odbm_fetch);
    DEFPRIM("odbm-delete", tc_subr_1, stk_odbm_delete);
    DEFPRIM("odbm-firstkey", tc_subr_0, stk_odbm_firstkey);
    DEFPRIM("odbm-nextkey", tc_subr_1, stk_odbm_nextkey);
}
