/*
 * ndbm - STk-ndbm bridge
 *
 *  (c) 1999 Shiro Kawai  shiro@acm.org
 *  Terms and conditions to use, modify or distribute this code
 *  is the same as of STk.
 *
 *  $Id: ndbm.c,v 1.4 2000/02/12 07:08:21 shiro Exp $
 */

#include <stk.h>
#include <fcntl.h>

#ifdef HAVE_NDBM_H
#include <ndbm.h>
#elif HAVE_GDBM_NDBM_H
#include <gdbm/ndbm.h>
#endif

static int tc_ndbm;

typedef struct {
    SCM name;
    DBM *dbf;                   /* NULL if closed */
} STk_ndbm;

#define NDBMP(obj)           TYPEP(obj, tc_ndbm)
#define NDBM_DATA(obj)       ((STk_ndbm*)EXTDATA(obj))
#define NDBM_DBF(obj)        NDBM_DATA(obj)->dbf

static void ndbm_mark(SCM);
static void ndbm_free(SCM);
static void ndbm_display(SCM, SCM, int);

static STk_extended_scheme_type STk_ndbm_typedesc = {
    "ndbm-file",                /* name */
    0,                          /* procp */
    ndbm_mark,                  /* mark */
    ndbm_free,                  /* free */
    NULL,                       /* apply */
    ndbm_display,               /* display */
    NULL                        /* compare */
};

static void ndbm_mark(SCM obj)
{
    STk_gc_mark(NDBM_DATA(obj)->name);
}

static void ndbm_free(SCM obj)
{
    STk_ndbm *data = NDBM_DATA(obj);
    if (data->dbf) dbm_close(data->dbf);
    free(data);
}

static void ndbm_display(SCM obj, SCM port, int mode)
{
    Puts("#<ndbm-file ", port);
    STk_print(NDBM_DATA(obj)->name, port, mode);
    Puts(">", port);
}

static SCM ndbm_new(SCM sname, int flags, int mode)
{
    SCM z;
    STk_ndbm *data;
    char *name = CHARS(sname);  /* type ensured by caller */
    DBM *dbf = dbm_open(name, flags, mode);
    if (dbf == NULL) return UNDEFINED; /* error handled by caller */
    data = (STk_ndbm*)must_malloc(sizeof(STk_ndbm));
    data->name = sname;
    data->dbf = dbf;
    NEWCELL(z, tc_ndbm);
    EXTDATA(z) = data;
    return z;
}

/* 
 * ndbm-open NAME &optional SIZE RWMODE FMODE ERRORCB
 */

PRIMITIVE stk_ndbm_open(SCM sname, SCM sflags, SCM smode)
{
    SCM sndbm;
    ENTER_PRIMITIVE("ndbm-open");

    if (!STRINGP(sname)) Serror("string required", sname);
    if (!INTEGERP(sflags)) Serror("integer required", sflags);
    if (!INTEGERP(smode)) Serror("integer required", smode);
    sndbm = ndbm_new(sname, INTEGER(sflags), INTEGER(smode));
    if (sndbm == UNDEFINED) {
        Serror("couldn't open ndbm database", sname);
    }
    return sndbm;
}

/* ndbm-close NDBM
 */

PRIMITIVE stk_ndbm_close(SCM obj)
{
    ENTER_PRIMITIVE("ndbm-close");
    if (!NDBMP(obj)) Serror("ndbm-file required", obj);
    if (NDBM_DBF(obj)) {
        dbm_close(NDBM_DBF(obj));
        NDBM_DBF(obj) = NULL;
    }
    return UNDEFINED;
}

/* ndbm-closed? NDBM
 */

PRIMITIVE stk_ndbm_closed(SCM obj)
{
    ENTER_PRIMITIVE("ndbm-closed?");
    if (!NDBMP(obj)) Serror("ndbm-file required", obj);
    return NDBM_DBF(obj)? Ntruth : Truth;
}

/* ndbm-store NDBM KEY CONTENT &optional FLAG
 */

PRIMITIVE stk_ndbm_store(SCM args, int argc)
{
    SCM sndbm, skey, sval;
    int flags = 0;
    int r;
    datum key, val;
    ENTER_PRIMITIVE("ndbm-store");

    if (argc < 3 || argc > 4) Serror("wrong # of args", args);
    sndbm = CAR(args); args = CDR(args);
    skey = CAR(args); args = CDR(args);
    sval = CAR(args); args = CDR(args);
    if (CONSP(args)) {
        SCM f = CAR(args);
        if (!EXACTP(f)) Serror("exact number is required for FLAGS", f);
        flags = STk_integer_value(f);
    }
    if (!NDBMP(sndbm)) Serror("ndbm-file required", sndbm);
    if (!STRINGP(skey)) Serror("string is required for a key", skey);
    if (!STRINGP(sval)) Serror("string is required for a val", sval);
    if (!NDBM_DBF(sndbm)) Serror("ndbm-file already closed", sndbm);
    key.dptr  = CHARS(skey);
    key.dsize = STRSIZE(skey);
    val.dptr  = CHARS(sval);
    val.dsize = STRSIZE(sval);
    r = dbm_store(NDBM_DBF(sndbm), key, val, flags);
    return STk_makeinteger(r);
}

/* ndbm-fetch NDBM KEY
 */

PRIMITIVE stk_ndbm_fetch(SCM sndbm, SCM skey)
{
    datum key, val;
    SCM r;
    ENTER_PRIMITIVE("ndbm-fetch");
    if (!NDBMP(sndbm)) Serror("ndbm-file required", sndbm);
    if (!STRINGP(skey)) Serror("string is required for a key", skey);
    if (!NDBM_DBF(sndbm)) Serror("ndbm-file already closed", sndbm);
    key.dptr  = CHARS(skey);
    key.dsize = STRSIZE(skey);
    val = dbm_fetch(NDBM_DBF(sndbm), key);
    if (val.dptr) {
        r = STk_makestrg(val.dsize, val.dptr);
        /* no need to free val.dptr: ndbm does that. */
    } else {
        r = Ntruth;
    }
    return r;
}

/* ndbm-delete NDBM KEY
 */

PRIMITIVE stk_ndbm_delete(SCM sndbm, SCM skey)
{
    datum key;
    int r;
    ENTER_PRIMITIVE("ndbm-delete");
    if (!NDBMP(sndbm)) Serror("ndbm-file required", sndbm);
    if (!STRINGP(skey)) Serror("string is required for a key", skey);
    if (!NDBM_DBF(sndbm)) Serror("ndbm-file already closed", sndbm);
    key.dptr  = CHARS(skey);
    key.dsize = STRSIZE(skey);
    r = dbm_delete(NDBM_DBF(sndbm), key);
    return STk_makeinteger(r);
}

/* ndbm-firstkey NDBM
 */

PRIMITIVE stk_ndbm_firstkey(SCM sndbm)
{
    datum key;
    SCM r;
    ENTER_PRIMITIVE("ndbm-firstkey");
    if (!NDBMP(sndbm)) Serror("ndbm-file required", sndbm);
    if (!NDBM_DBF(sndbm)) Serror("ndbm-file already closed", sndbm);
    key = dbm_firstkey(NDBM_DBF(sndbm));
    if (key.dptr) {
        r = STk_makestrg(key.dsize, key.dptr);
        /* no need to free key.dptr: ndbm does that. */
    } else {
        r = Ntruth;
    }
    return r;
}

/* ndbm-nextkey NDBM KEY
 */

PRIMITIVE stk_ndbm_nextkey(SCM sndbm)
{
    datum nkey;
    SCM r;
    ENTER_PRIMITIVE("ndbm-nextkey");
    if (!NDBMP(sndbm)) Serror("ndbm-file required", sndbm);
    if (!NDBM_DBF(sndbm)) Serror("ndbm-file already closed", sndbm);
    nkey = dbm_nextkey(NDBM_DBF(sndbm));
    if (nkey.dptr) {
        r = STk_makestrg(nkey.dsize, nkey.dptr);
        /* no need to free nkey.dptr: ndbm does that. */
    } else {
        r = Ntruth;
    }
    return r;
}

/* ndbm-error NDBM
 */

PRIMITIVE stk_ndbm_error(SCM sndbm)
{
    int r;
    ENTER_PRIMITIVE("ndbm-error");
    if (!NDBMP(sndbm)) Serror("ndbm-file required", sndbm);
    if (!NDBM_DBF(sndbm)) Serror("ndbm-file already closed", sndbm);
    r = dbm_error(NDBM_DBF(sndbm));
    return STk_makeinteger(r);
}
/* ndbm-clearerror NDBM
 */

PRIMITIVE stk_ndbm_clear_error(SCM sndbm)
{
    int r;
    ENTER_PRIMITIVE("ndbm-clear-error");
    if (!NDBMP(sndbm)) Serror("ndbm-file required", sndbm);
    if (!NDBM_DBF(sndbm)) Serror("ndbm-file already closed", sndbm);
    dbm_clearerr(NDBM_DBF(sndbm));
    return UNDEFINED;
}

/* Initialization */

#define DEFCONST(var) \
    STk_define_variable(#var, STk_makeinteger(var), STk_get_selected_module())

#define DEFPRIM(name, tag, proc) \
    STk_add_new_primitive(name, tag, (PRIMITIVE(*)())proc)

void STk_init_ndbm()
{
    tc_ndbm = STk_add_new_type(&STk_ndbm_typedesc);
    DEFPRIM("ndbm-open", tc_subr_3, stk_ndbm_open);
    DEFPRIM("ndbm-close", tc_subr_1, stk_ndbm_close);
    DEFPRIM("ndbm-closed?", tc_subr_1, stk_ndbm_closed);
    DEFPRIM("ndbm-store", tc_lsubr, stk_ndbm_store);
    DEFPRIM("ndbm-fetch", tc_subr_2, stk_ndbm_fetch);
    DEFPRIM("ndbm-delete", tc_subr_2, stk_ndbm_delete);
    DEFPRIM("ndbm-firstkey", tc_subr_1, stk_ndbm_firstkey);
    DEFPRIM("ndbm-nextkey", tc_subr_1, stk_ndbm_nextkey);
    DEFPRIM("ndbm-error", tc_subr_1, stk_ndbm_error);
    DEFPRIM("ndbm-clear-error", tc_subr_1, stk_ndbm_clear_error);


    /* flags for dbm-store */
    DEFCONST(DBM_INSERT);
    DEFCONST(DBM_REPLACE);

    /* flags for dbm-open */
    DEFCONST(O_RDONLY);
    DEFCONST(O_WRONLY);
    DEFCONST(O_RDWR);
    DEFCONST(O_CREAT);
    DEFCONST(O_TRUNC);
}
