/*
 * gdbm - STk-gdbm bridge
 *
 *  (c) 1999 Shiro Kawai  shiro@acm.org
 *  Terms and conditions to use, modify or distribute this code
 *  is the same as of STk.
 *
 *  $Id: gdbm.c,v 1.3 2000/02/12 07:08:21 shiro Exp $
 */

#include <stk.h>
#include <gdbm.h>

static int tc_gdbm;

typedef struct {
    SCM name;
    GDBM_FILE dbf;              /* NULL if closed */
} STk_gdbm;

#define GDBMP(obj)           TYPEP(obj, tc_gdbm)
#define GDBM_DATA(obj)       ((STk_gdbm*)EXTDATA(obj))
#define GDBM_DBF(obj)        GDBM_DATA(obj)->dbf

static void gdbm_mark(SCM);
static void gdbm_free(SCM);
static void gdbm_display(SCM, SCM, int);

static STk_extended_scheme_type STk_gdbm_typedesc = {
    "gdbm-file",                /* name */
    0,                          /* procp */
    gdbm_mark,                  /* mark */
    gdbm_free,                  /* free */
    NULL,                       /* apply */
    gdbm_display,               /* display */
    NULL                        /* compare */
};

static void gdbm_mark(SCM obj)
{
    STk_gc_mark(GDBM_DATA(obj)->name);
}

static void gdbm_free(SCM obj)
{
    STk_gdbm *data = GDBM_DATA(obj);
    if (data->dbf) gdbm_close(data->dbf);
    free(data);
}

static void gdbm_display(SCM obj, SCM port, int mode)
{
    Puts("#<gdbm-file", port);
    STk_print(GDBM_DATA(obj)->name, port, mode);
    Puts(">", port);
}

static SCM gdbm_new(SCM sname, int bsize, int rwmode, int fmode, void (*error_cb)())
{
    SCM z;
    STk_gdbm *data;
    char *name = CHARS(sname);  /* type ensured by caller */
    GDBM_FILE dbf = gdbm_open(name, bsize, rwmode, fmode, error_cb);
    if (dbf == NULL) return UNDEFINED; /* error handled by caller */
    data = (STk_gdbm*)must_malloc(sizeof(STk_gdbm));
    data->name = sname;
    data->dbf = dbf;
    NEWCELL(z, tc_gdbm);
    EXTDATA(z) = data;
    return z;
}

/* 
 * gdbm-open NAME &optional SIZE RWMODE FMODE ERRORCB
 */

PRIMITIVE stk_gdbm_open(SCM args, int argc)
{
    SCM sname;
    int size = 0;
    int rwmode = GDBM_READER;
    int fmode = 0666;
    SCM sproc = UNDEFINED;
    SCM sgdbm;
    ENTER_PRIMITIVE("gdbm-open");

    if (argc < 1) Serror("too few args", NIL);
    sname = CAR(args); args = CDR(args);
    if (!STRINGP(sname)) Serror("string required", sname);
    if (CONSP(args)) {
        SCM ssize = CAR(args); args = CDR(args);
        if (!EXACTP(ssize)) Serror("exact number required", ssize);
        size = STk_integer_value(ssize);
        if (CONSP(args)) {
            SCM srwmode = CAR(args); args = CDR(args);
            if (!EXACTP(srwmode)) Serror("exact number required", srwmode);
            rwmode = STk_integer_value(srwmode);
            if (CONSP(args)) {
                SCM sfmode = CAR(args); args = CDR(args);
                if (!EXACTP(sfmode)) Serror("exact number required", sfmode);
                fmode = STk_integer_value(sfmode);
                if (CONSP(args)) {
                    sproc = CAR(args); args = CDR(args);
                    if (!STk_procedurep(sproc))
                        Serror("procedure required", sproc);
                    if (CONSP(args)) {
                        Serror("too many arguments", args);
                    }
                }
            }
        }
    }
    sgdbm = gdbm_new(sname, size, rwmode, fmode, NULL);
    if (sgdbm == UNDEFINED) {
        Serror("couldn't open gdbm database", sname);
    }
    return sgdbm;
}

/* gdbm-close GDBM
 */

PRIMITIVE stk_gdbm_close(SCM obj)
{
    ENTER_PRIMITIVE("gdbm-close");
    if (!GDBMP(obj)) Serror("gdbm-file required", obj);
    if (GDBM_DBF(obj)) {
        gdbm_close(GDBM_DBF(obj));
        GDBM_DBF(obj) = NULL;
    }
    return UNDEFINED;
}

/* gdbm-closed? GDBM
 */

PRIMITIVE stk_gdbm_closed(SCM obj)
{
    ENTER_PRIMITIVE("gdbm-closed?");
    if (!GDBMP(obj)) Serror("gdbm-file required", obj);
    return GDBM_DBF(obj)? Ntruth : Truth;
}

/* gdbm-store GDBM KEY CONTENT &optional FLAG
 */

PRIMITIVE stk_gdbm_store(SCM args, int argc)
{
    SCM sgdbm, skey, sval;
    int flags = 0;
    int r;
    datum key, val;
    ENTER_PRIMITIVE("gdbm-store");

    if (argc < 3 || argc > 4) Serror("wrong # of args", args);
    sgdbm = CAR(args); args = CDR(args);
    skey = CAR(args); args = CDR(args);
    sval = CAR(args); args = CDR(args);
    if (CONSP(args)) {
        SCM f = CAR(args);
        if (!EXACTP(f)) Serror("exact number is required for FLAGS", f);
        flags = STk_integer_value(f);
    }
    if (!GDBMP(sgdbm)) Serror("gdbm-file required", sgdbm);
    if (!STRINGP(skey)) Serror("string is required for a key", skey);
    if (!STRINGP(sval)) Serror("string is required for a val", sval);
    if (!GDBM_DBF(sgdbm)) Serror("gdbm-file already closed", sgdbm);
    key.dptr  = CHARS(skey);
    key.dsize = STRSIZE(skey);
    val.dptr  = CHARS(sval);
    val.dsize = STRSIZE(sval);
    r = gdbm_store(GDBM_DBF(sgdbm), key, val, flags);
    return STk_makeinteger(r);
}

/* gdbm-fetch GDBM KEY
 */

PRIMITIVE stk_gdbm_fetch(SCM sgdbm, SCM skey)
{
    datum key, val;
    SCM r;
    ENTER_PRIMITIVE("gdbm-fetch");
    if (!GDBMP(sgdbm)) Serror("gdbm-file required", sgdbm);
    if (!STRINGP(skey)) Serror("string is required for a key", skey);
    if (!GDBM_DBF(sgdbm)) Serror("gdbm-file already closed", sgdbm);
    key.dptr  = CHARS(skey);
    key.dsize = STRSIZE(skey);
    val = gdbm_fetch(GDBM_DBF(sgdbm), key);
    if (val.dptr) {
        r = STk_makestrg(val.dsize, val.dptr);
        free(val.dptr);
    } else {
        r = Ntruth;
    }
    return r;
}

/* gdbm-delete GDBM KEY
 */

PRIMITIVE stk_gdbm_delete(SCM sgdbm, SCM skey)
{
    datum key;
    int r;
    ENTER_PRIMITIVE("gdbm-delete");
    if (!GDBMP(sgdbm)) Serror("gdbm-file required", sgdbm);
    if (!STRINGP(skey)) Serror("string is required for a key", skey);
    if (!GDBM_DBF(sgdbm)) Serror("gdbm-file already closed", sgdbm);
    key.dptr  = CHARS(skey);
    key.dsize = STRSIZE(skey);
    r = gdbm_delete(GDBM_DBF(sgdbm), key);
    return STk_makeinteger(r);
}

/* gdbm-firstkey GDBM
 */

PRIMITIVE stk_gdbm_firstkey(SCM sgdbm)
{
    datum key;
    SCM r;
    ENTER_PRIMITIVE("gdbm-firstkey");
    if (!GDBMP(sgdbm)) Serror("gdbm-file required", sgdbm);
    if (!GDBM_DBF(sgdbm)) Serror("gdbm-file already closed", sgdbm);
    key = gdbm_firstkey(GDBM_DBF(sgdbm));
    if (key.dptr) {
        r = STk_makestrg(key.dsize, key.dptr);
        free(key.dptr);
    } else {
        r = Ntruth;
    }
    return r;
}

/* gdbm-nextkey GDBM KEY
 */

PRIMITIVE stk_gdbm_nextkey(SCM sgdbm, SCM skey)
{
    datum key, nkey;
    SCM r;
    ENTER_PRIMITIVE("gdbm-nextkey");
    if (!GDBMP(sgdbm)) Serror("gdbm-file required", sgdbm);
    if (!STRINGP(skey)) Serror("string is required for a key", skey);
    if (!GDBM_DBF(sgdbm)) Serror("gdbm-file already closed", sgdbm);
    key.dsize = STRSIZE(skey);
    key.dptr  = CHARS(skey);
    nkey = gdbm_nextkey(GDBM_DBF(sgdbm), key);
    if (nkey.dptr) {
        r = STk_makestrg(nkey.dsize, nkey.dptr);
        free(nkey.dptr);
    } else {
        r = Ntruth;
    }
    return r;
}

/* gdbm-reorganize GDBM
 */

PRIMITIVE stk_gdbm_reorganize(SCM sgdbm)
{
    int r;
    ENTER_PRIMITIVE("gdbm-reorganize");
    if (!GDBMP(sgdbm)) Serror("gdbm-file required", sgdbm);
    if (!GDBM_DBF(sgdbm)) Serror("gdbm-file already closed", sgdbm);
    r = gdbm_reorganize(GDBM_DBF(sgdbm));
    return STk_makeinteger(r);
}

/* gdbm-sync GDBM
 */

PRIMITIVE stk_gdbm_sync(SCM sgdbm)
{
    ENTER_PRIMITIVE("gdbm-sync");
    if (!GDBMP(sgdbm)) Serror("gdbm-file required", sgdbm);
    if (!GDBM_DBF(sgdbm)) Serror("gdbm-file already closed", sgdbm);
    gdbm_sync(GDBM_DBF(sgdbm));
    return UNDEFINED;
}

/* gdbm-exists GDBM KEY
 */

PRIMITIVE stk_gdbm_exists(SCM sgdbm, SCM skey)
{
    datum key;
    int r;
    ENTER_PRIMITIVE("gdbm-exists");
    if (!GDBMP(sgdbm)) Serror("gdbm-file required", sgdbm);
    if (!STRINGP(skey)) Serror("string is required for a key", skey);
    if (!GDBM_DBF(sgdbm)) Serror("gdbm-file already closed", sgdbm);
    key.dsize = STRSIZE(skey);
    key.dptr  = CHARS(skey);
    r = gdbm_exists(GDBM_DBF(sgdbm), key);
    return r ? Truth : Ntruth;
}

/* gdbm-strerror ERRNO
 */

PRIMITIVE stk_gdbm_strerror(SCM serrno)
{
    char *str;
    ENTER_PRIMITIVE("gdbm-strerror");
    if (!INTEGERP(serrno)) Serror("exact integer required", serrno);
    str = gdbm_strerror(INTEGER(serrno));
    return STk_makestring(str);
}

/* gdbm-setopt GDBM OPTION VAL SIZE
 */

PRIMITIVE stk_gdbm_setopt(SCM sgdbm, SCM sopt, SCM sval)
{
    int val, r;
    ENTER_PRIMITIVE("gdbm-setopt");
    if (!GDBMP(sgdbm)) Serror("gdbm-file required", sgdbm);
    if (!GDBM_DBF(sgdbm)) Serror("gdbm-file already closed", sgdbm);
    if (!INTEGERP(sopt)) Serror("integer required", sopt);
    if (!INTEGERP(sval) && !BOOLP(sval)) Serror("integer or boolean required",
                                                sval);
    if (INTEGERP(sval)) val = INTEGER(sval);
    else if (BOOLP(sval)) val = (sval == Ntruth)? FALSE : TRUE;
    r = gdbm_setopt(GDBM_DBF(sgdbm), INTEGER(sopt), &val, sizeof(int));
    return STk_makeinteger(r);
}

/* gdbm-version
 */

PRIMITIVE stk_gdbm_version()
{
    return STk_makestring(gdbm_version);
}

/* *gdbm-errno* accessor
 */

void set_gdbm_errno(SCM val)
{
    if (INTEGERP(val)) gdbm_errno = INTEGER(val);
    else gdbm_errno = 0;
}

SCM get_gdbm_errno()
{
    return STk_makeinteger(gdbm_errno);
}

/* Initialization */

#define DEFCONST(var) \
    STk_define_variable(#var, STk_makeinteger(var), STk_get_selected_module())

#define DEFPRIM(name, tag, proc) \
    STk_add_new_primitive(name, tag, (PRIMITIVE(*)())proc)

void STk_init_gdbm()
{
    tc_gdbm = STk_add_new_type(&STk_gdbm_typedesc);
    DEFPRIM("gdbm-open", tc_lsubr, stk_gdbm_open);
    DEFPRIM("gdbm-close", tc_subr_1, stk_gdbm_close);
    DEFPRIM("gdbm-closed?", tc_subr_1, stk_gdbm_closed);
    DEFPRIM("gdbm-store", tc_lsubr, stk_gdbm_store);
    DEFPRIM("gdbm-fetch", tc_subr_2, stk_gdbm_fetch);
    DEFPRIM("gdbm-delete", tc_subr_2, stk_gdbm_delete);
    DEFPRIM("gdbm-firstkey", tc_subr_1, stk_gdbm_firstkey);
    DEFPRIM("gdbm-nextkey", tc_subr_2, stk_gdbm_nextkey);
    DEFPRIM("gdbm-reorganize", tc_subr_1, stk_gdbm_reorganize);
    DEFPRIM("gdbm-sync", tc_subr_1, stk_gdbm_sync);
    DEFPRIM("gdbm-exists", tc_subr_2, stk_gdbm_exists);
    DEFPRIM("gdbm-strerror", tc_subr_1, stk_gdbm_strerror);
    DEFPRIM("gdbm-setopt", tc_subr_3, stk_gdbm_setopt);
    DEFPRIM("gdbm-version", tc_subr_0, stk_gdbm_version);

    STk_define_C_variable("*gdbm-errno*",
                          get_gdbm_errno,
                          (void(*)())set_gdbm_errno);

    /* flags for gdbm-open */
    DEFCONST(GDBM_READER);
    DEFCONST(GDBM_WRITER);
    DEFCONST(GDBM_WRCREAT);
    DEFCONST(GDBM_NEWDB);
    DEFCONST(GDBM_FAST);
    DEFCONST(GDBM_SYNC);
    DEFCONST(GDBM_NOLOCK);

    /* Parameters to gdbm_store */
    DEFCONST(GDBM_INSERT);
    DEFCONST(GDBM_REPLACE);

    /* Parameters to gdbm_setopt */
    DEFCONST(GDBM_CACHESIZE);
    DEFCONST(GDBM_FASTMODE);
    DEFCONST(GDBM_SYNCMODE);
    DEFCONST(GDBM_CENTFREE);
    DEFCONST(GDBM_COALESCEBLKS);
}
